/*
   This program starts a child process and wants the child
   to inherit (share) this program's stdin and stdout streams.
   But this cannot be done in Java 6 (or earlier). So we fake
   it (in a complicated way).

   In this program, two independent "threads" are created.
   One thread "pumps" data from the parent's stdin to the
   child's stdin. The other thread "pumps" data from the
   child's stdout to the parent's stdout. These two threads
   (which act as pumps between streams) are replacements for
   the "pipes" that should be connecting the streams, but
   which Java does not provide.


   The structure of this program is similar to the program
   Java6_RedirectChildStdinStdoutToFiles_ver3.java


   Important: Notice that this program is logically the
   same as the following command line.

   C:\> java Double

   In the case of the above command line, the shell
   program (cmd.exe) runs the program Double.class
   and has it inherit the shell's stdin and stdout.
   In the case of this program, this program itself
   runs Double.class and makes sure that Double.class
   inherit's this program's stdin and stdout. So this
   program is acting like a very simple shell program.
*/
import java.util.Scanner;
import java.io.*;

public class Java6_ChildInheritsStdStreams_ver3
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Create a command line for running the child.
      String cmd = "java Double";
      // Execute the filter.
      Process process = Runtime.getRuntime().exec(cmd);

      // Create a stream that writes data to the child's stdin stream.
      final PrintStream stdinOfChild =  new PrintStream( process.getOutputStream() );
      // Create a stream that reads data from the child's stdout stream.
      final Scanner stdoutOfChild = new Scanner( process.getInputStream() );

      // Create a Scanner object to make it easier to use System.in
      final Scanner scanner = new Scanner( System.in );


      // Create a "pump" between the parent's stdin
      // and the child's input stream.
      new Thread(new Runnable(){public void run()
      {while(scanner.hasNextLine())
       {String oneLine = scanner.nextLine();   // read from parent's stdin
        stdinOfChild.println( oneLine );       // write to child's stdin
        //stdinOfChild.flush();                // SEE THE REMARKS BELOW
       }stdinOfChild.close(); //IMPORTANT (otherwise the child hangs)
      }}).start();


      // Create a "pump" between the child's output stream
      // and the parent's stdout.
      new Thread(new Runnable(){public void run()
      {while(stdoutOfChild.hasNextLine())
       {String oneLine = stdoutOfChild.nextLine(); // read from child's stdout
        System.out.println( oneLine );             // write to the parent's stdout
      }}}).start();

      // Wait for the child to finish its work.
      process.waitFor();  // this throws InterruptedException
   }
}

/*
   IMPORTANT:
   Line 60,
      stdinOfChild.flush();
   is commented out because it is not needed for this program to
   be correct. However, this line is needed for the program to
   behave in a nice way when stdin is the keyboard and stdout
   is the console screen.

   Do the following experiment. Run this program with stdin and
   stdout being the usual, default console window. When you type
   input lines and hit ENTER, you will not get the doubled result
   echoed to you. When you finally terminate the input (with a ^Z)
   all the echoed output will appear at once. Notice that this is
   not really incorrect behavior, it is just kind of unexpected
   (though some Linux command line programs do act this way).

   Now uncomment out line 46, recompile the program, and redo the
   experiment. Now each time you hit ENTER at the end of an input
   line, the doubled result is echoed to you immediately.

   The flush() method tells any buffer holding the results from
   println() to immediately send what is held in the buffer to
   the output destination. For efficiency reasons, the buffer's
   default behavior is to hold all of its results until the
   buffer is full, and then send the buffer contents to its
   destination (and when stdin and stdout are files, this default
   behavior works very well). When this program is reading from
   the keyboard, without the flush() method you would have to type
   at the keyboard enough characters to fill an output buffer
   before you see anything echoed to the console window. But the
   buffers are tens of thousands of bytes in size, so that is a
   lot of typing.

   In the case of the stdin and stdout being the console window,
   it is better to flush every line of input. But in the case of
   stdin and stdout being files, it is better not to flush every
   input line. This is one of the complexities of working with
   standard streams.
*/